/**
 * login 模块
 * @module login
 * @class login
 */
define([ 'jquery', 'knockout', 'config/config', 'service', 'underscore', 'config/menu', "logout" ],
    function ($, ko, config, service, _, menu, logout) {

        var pageState = {LOGIN:0, WAIT_PIN:1, WAIT_PUK:2, PUK_LOCKED:3, LOGGEDIN:4, LOADING:5};
        var timer = startLoginStatusInterval();

        function startLoginStatusInterval() {
            return setInterval(function () {
                var info = service.getStatusInfo();
                if (!info.isLoggedIn) {
                    gotoLogin();
                    return;
                }

                lastLoginStatus = service.getStatusInfo().isLoggedIn ? "1" : "0";
            }, 1000);
        }

        /**
         * loginViewModel
         * @class loginVM
         */
        function loginVM() {
            var self = this;

            self.adImages = config.AD_IMAGES;

            var data = service.getLoginData();
            var loginStatus = service.getLoginStatus();
            self.password = ko.observable();
            self.PIN = ko.observable();
            self.PUK = ko.observable();
            self.newPIN = ko.observable();
            self.confirmPIN = ko.observable();
            self.pinNumber = ko.observable(data.pinnumber);
            self.pukNumber = ko.observable(data.puknumber);
            self.showEntrance = ko.observable(false);

            service.getSDConfiguration({}, function (data) {
                self.showEntrance(data.sd_status == "1" && data.share_status == "1" && data.sd_mode == "0");
            });

            var state = computePageState(loginStatus, data);
            self.pageState = ko.observable(state);
            if (state == pageState.LOADING) {
                addTimeout(refreshPage, 500);
            }

            /**
             * login 事件处理
             * @event login
             */
            self.login = function () {
                self.pageState(pageState.LOADING);
                window.clearInterval(timer);
                service.login({
                    password:self.password()
                }, function (data) {
                    setTimeout(function () {
                        timer = startLoginStatusInterval();
                    }, 1300);
                    if (data.result) {
                        self.pageState(pageState.LOGGEDIN);
                        window.location.hash = "#home";
                        menu.rebuild();
                        logout.init();
                    } else {
                        self.password("");
                        showAlert("password_error", function () {
                            $("#txtPwd").focus();
                        });
                        self.pageState(pageState.LOGIN);
                    }
                });
            };

            /**
             * 验证输入PIN事件处理
             *
             * @event enterPIN
             */
            self.enterPIN = function () {
                self.pageState(pageState.LOADING);
                var pin = self.PIN();
                service.enterPIN({
                    PinNumber:pin
                }, function (data) {
                    if (!data.result) {
                        showAlert("pin_error", function () {
                            refreshPage();
                        });
                        self.PIN('');
                    } else {
                        refreshPage();
                    }
                });
            };

            /**
             * 输入PUK设置新PIN事件处理
             *
             * @event enterPUK
             */
            self.enterPUK = function () {
                self.pageState(pageState.LOADING);
                var newPIN = self.newPIN();
                var confirmPIN = self.confirmPIN();
                var params = {};
                params.PinNumber = newPIN;
                params.PUKNumber = self.PUK();
                service.enterPUK(params, function (data) {
                    if (!data.result) {
                        showAlert("puk_error", function () {
                            refreshPage();
                        });
                        self.PUK('');
                        self.newPIN('');
                        self.confirmPIN('');
                    } else {
                        refreshPage();
                    }
                });
            };
            /**
             * 刷新页面状态
             *
             * @method refreshPage
             */
            function refreshPage() {
                var data = service.getLoginData();
                var loginStatus = service.getLoginStatus();
                var state = computePageState(loginStatus, data);
                if (state == pageState.LOADING) {
                    addTimeout(refreshPage, 500);
                } else {
                    self.pageState(state);
                    self.pinNumber(data.pinnumber);
                    self.pukNumber(data.puknumber);
                }
            }

            /**
             * 根据登录状态和SIM卡状态设置页面状态
             * @method computePageState
             */
            function computePageState(loginStatus, data) {
                if (loginStatus.status == "loggedIn") {
                    return pageState.LOGGEDIN;
                } else {
                    var state = data.modem_main_state;
                    if ($.inArray(state, config.TEMPORARY_MODEM_MAIN_STATE) != -1) {
                        return pageState.LOADING;
                    } else if (state == "modem_waitpin") {
                        return pageState.WAIT_PIN;
                    } else if ((state == "modem_waitpuk" || data.pinnumber == 0) && (data.puknumber != 0)) {
                        return pageState.WAIT_PUK;
                    } else if ((data.puknumber == 0 || state == "modem_sim_destroy")
                        && state != "modem_sim_undetected" && state != "modem_undetected") {
                        return pageState.PUK_LOCKED;
                    } else {
                        return pageState.LOGIN;
                    }
                }
            }
        }

        /**
         * 登录页背景高度自适应
         * @method resizeLoginPage
         */
        function resizeLoginPage() {
            if (window.location.hash != '#login') {
                return false;
            }
            var mainContainer = $("#mainContainer");
            var windowH = $(window).height();
            var statusH = $("#topStatus").outerHeight(true);
            var footerH = $("#footer").outerHeight(true);
            var containerHeight = windowH - statusH - mainContainer.height() - footerH;
            if (containerHeight > 0) {
                var h = windowH - statusH - footerH - 5;
                mainContainer.height(h + 'px');
                var container = $("#container");
                var margin = (h - container.height()) / 2;
                container.css({'margin-top':margin + 'px', 'margin-bottom':margin + 'px'});
            }
        }

        /**
         * 初始化ViewModel
         *
         * @method init
         */
        function init() {
            var info = service.getStatusInfo();
            if (info.isLoggedIn) {
                window.location.hash = '#home';
                return;
            }

            var container = $('#container')[0];
            ko.cleanNode(container);
            var vm = new loginVM();
            ko.applyBindings(vm, container);
            resizeLoginPage();
            var throttleTimeout;
            $(window).bind('resize', function () {
                if ($.browser.msie) {
                    if (!throttleTimeout) {
                        throttleTimeout = setTimeout(function () {
                            resizeLoginPage();
                            throttleTimeout = null;
                        }, 50);
                    }
                } else {
                    resizeLoginPage();
                }
            });
            $('#frmLogin').validate({
                submitHandler:function () {
                    vm.login();
                }
            });
            $('#frmPIN').validate({
                submitHandler:function () {
                    vm.enterPIN();
                },
                rules:{
                    txtPIN:"pin_check"
                }
            });
            $('#frmPUK').validate({
                submitHandler:function () {
                    vm.enterPUK();
                },
                rules:{
                    txtNewPIN:"pin_check",
                    txtConfirmPIN:{equalTo:"#txtNewPIN"},
                    txtPUK:"puk_check"
                }
            });
        }

        function gotoLogin() {
            if (window.location.hash != config.defaultRoute && _.indexOf(config.GUEST_HASH, window.location.hash) == -1) {
                if (!manualLogout && lastLoginStatus == "1") {
                    manualLogout = false;
                    lastLoginStatus = 'UNREAL';
                    showAlert('need_login_again', function () {
                        window.location = "index.html";
                    });
                }
                else if (lastLoginStatus == 'UNREAL') {
                    //do nothing, only popup need_login_again alert one time
                    return;
                }
                else {
                    window.location = "index.html";
                }

            }
        }

        return {
            init:init,
            gotoLogin:gotoLogin
        };
    });